package org.sc3d.apt.jrider.v1;

import java.awt.event.*;

/** There are 31 keys that the game might respond to: 5 for each of 6 players, and a key to quit. Ideally, the game needs to know which combination of keys is pressed every millisecond. Java does not offer sufficient real-time guarantees to achieve this, and even if it did there would be no hope of achieving it in networked games. The best we can do is to record the time at which the game becomes aware of key-presses, which is a reasonable approximation of the time the key was pressed. That's what this class does. This class only handles a local game, but a subclass could implement the server side of a network game.
 * <p>The 31 logical keys are numbered from 0 to 30, as follows:<ul>
 * <li>0 is red's key to steer left.
 * <li>1 is red's key to steer right.
 * <li>2 is red's accelerator.
 * <li>3 is red's brake.
 * <li>4 is red's key to change gear.
 * <li>5 to 9 are green's keys.
 * <li>10 to 14 are yellow's keys.
 * <li>15 to 19 are blue's keys.
 * <li>20 to 24 are magenta's keys.
 * <li>25 to 29 are cyan's keys.
 * <li>30 is the quit button.
 * </ul>The physical keys can be redefined: the constructor takes an array of virtual key codes. Default definitions are provided for convenience.
 */
public class Controller implements KeyListener {
  /** Constructs a Controller, given the key definitions.
   * @param keyCodes an array of length 31 giving the virtual key code for each of the logical keys. Any entry may be '-1' to indicate that no key is defined.
   */
  public Controller(int[] keyCodes) {
    this.keyCodes = keyCodes;
    this.current = 0;
    this.lastTime = System.currentTimeMillis();
    this.used = 0;
  }
  
  /* New API. */
  
  /** Works out which, if any, of the 31 keys was involved in the specified KeyEvent.
   * @return a code from '0' to '30', or '-1'.
   */
  public int decode(KeyEvent e) {
    int code = e.getKeyCode();
    for (int i=0; i<31; i++) if (this.keyCodes[i]==code) return i;
    return -1;
  }
  
  /** Prints the key definitions to 'System.out'. */
  public void printKeys(int numCars) {
    for (int i=0; i<numCars; i++) {
      System.out.println("Car "+i+":");
      for (int j=0; j<5; j++) {
        System.out.println(
          "  "+ACTIONS[j]+":"+KeyEvent.getKeyText(this.keyCodes[i*5+j])
        );
      }
    }
    System.out.println("Quit:"+KeyEvent.getKeyText(this.keyCodes[30]));
  }
  
  /** Returns a bitmask for every millisecond that has passed since this method was last called. Bit 'n' in a mask indicates whether key 'n' was pressed in that millisecond. If there is more than a second of data, some may be discarded. */
  public synchronized int[] getKeyData() {
    this.fill();
    int[] ans = new int[this.used];
    System.arraycopy(this.BUF, 0, ans, 0, this.used);
    this.used = 0;
    return ans;
  }
  
  /** The names of the players. */
  public static final String[] PLAYERS = new String[] {
      "Red", "Green", "Yellow", "Blue", "Magenta", "Cyan"
  };
  
  /** The names of the five keys each player has. */
  public static final String[] ACTIONS = new String[] {
      "left", "right", "accelerate", "brake", "change gear"
  };
  
  /** Default key definitions, mainly for debugging. */
  public static final int[] DEFAULT_KEYS = new int[] {
    KeyEvent.VK_Z, KeyEvent.VK_X, KeyEvent.VK_F, KeyEvent.VK_C, KeyEvent.VK_G,
    KeyEvent.VK_COMMA, KeyEvent.VK_PERIOD, KeyEvent.VK_QUOTE, KeyEvent.VK_SLASH, KeyEvent.VK_NUMBER_SIGN,
    KeyEvent.VK_NUMPAD1, KeyEvent.VK_NUMPAD2, KeyEvent.VK_NUMPAD6, KeyEvent.VK_NUMPAD3, KeyEvent.VK_ADD,
    KeyEvent.VK_LEFT, KeyEvent.VK_RIGHT, KeyEvent.VK_UP, KeyEvent.VK_DOWN, KeyEvent.VK_CONTROL,
    KeyEvent.VK_Q, KeyEvent.VK_E, KeyEvent.VK_3, KeyEvent.VK_W, KeyEvent.VK_4,
    -1, -1, -1, -1, -1,
    27
  };
  
  /* Implement things in KeyListener. */
  
  /** Called by the AWT when a key is pressed. */
  public synchronized void keyPressed(KeyEvent e) {
    final int code = this.decode(e); if (code==-1) return;
    this.fill(); this.current |= 1<<code;
  }

  /** Called by the AWT when a key is released. */
  public synchronized void keyReleased(KeyEvent e) {
    final int code = this.decode(e); if (code==-1) return;
    this.fill(); this.current &= ~(1<<code);
  }
  
  /** Called by the AWT when a character is typed. We don't care. */
  public void keyTyped(KeyEvent e) {}

  /* Private. */
  
  private void fill() {
    long time = System.currentTimeMillis();
    int newUsed = this.used + (int)(time-this.lastTime);
    if (newUsed>this.BUF.length) newUsed = this.BUF.length;
    while (this.used<newUsed) this.BUF[this.used++] = this.current;
    this.lastTime = time;
  }
  
  private int[] keyCodes;
  
  private int current;
  private long lastTime;
  private final int[] BUF = new int[1000];
  private int used;
  
  /* Test code. */
  public static void main(String[] args) {
    if (args.length!=0) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Controller"
    );
    final Controller me = new Controller(Controller.DEFAULT_KEYS);
    final java.awt.Frame frame = new java.awt.Frame("Testing Controller");
    frame.addKeyListener(me);
    frame.show();
    frame.requestFocus();
    while (true) {
      final int[] keyData = me.getKeyData();
      if (keyData.length>0) {
        System.out.println(
          "keyData[0] = "+Integer.toHexString(keyData[0])+", "+
          "keyData.length = "+keyData.length
        );
        if ((keyData[0]&(1<<30))!=0) System.exit(0);
      }
    }
  }
}
