package org.sc3d.apt.jrider.v1;

/** A Face represents something that needs to be drawn on top of the landscape, such as a triangle. This abstract superclass knows the bounding rectangle of the Face on the screen, and the distance to the face. Subclasses need only override one method: 'draw()'.
 * <p>Faces can organise themselves into a priority queue (the algorithm used is a leftist heap). The queue is ordered by left coordinate first (since the scene is drawn from left to right) and entries with the same left coordinate are ordered by distance (for painter's algorithm). 'null' represents the empty queue. A non-empty queue is specified by giving the first Face in the queue. A Face can only belong to one queue.
 */
public abstract class Face {
  /** Constructs a Face.
   * @param l the left coordinate in pixels.
   * @param t the top coordinate in pixels.
   * @param r the right coordinate in pixels.
   * @param b the bottom coordinate in pixels.
   * @param order the plot order of the Face. This must be one of the 'ORDER_XXX' values. Faces with smaller 'order' values get plotted first.
   * @param dist the distance from the viewer to the Face, in units such that the size of the landscape is '1&lt;&lt;16'. Faces with larger 'dist' values get plotted first, if their 'order' values are the same.
   * @param colour the colour of the Face, or '0xff' for a Face that looks like a shadow.
   */
  public Face(int l, int t, int r, int b, int order, int dist, byte colour) {
    this.l = l; this.t = t; this.r = r; this.b = b;
    this.order = order; this.dist = dist; this.colour = colour;
    this.big = this.small = null;
    this.depth = 1;
  }
  
  /* New API. */
  
  /** A value for the 'order' field, indicating that this Face represents paint on the landscape and should be drawn before shadows and objects. */
  public static final int ORDER_PAINT = 1;
  
  /** A value for the 'order' field, indicating that this Face represents a shadow on the landscape and should be drawn after paint and before objects. */
  public static final int ORDER_SHADOW = 2;
  
  /** A value for the 'order' field, indicating that this Face represents on object in front of the landscape and should be drawn after paint and shadows. */
  public static final int ORDER_OBJECT = 3;
  
  /** Merges the queue starting with 'this' with the queue starting with 'that' and returns the result. */
  public Face merge(Face that) {
    if (that==null) return this;
    if (
      this.l<that.l || (
        this.l==that.l && (this.order<that.order || (
          this.order==that.order && this.dist>that.dist
    )))) {
      return this.merge2(that);
    } else {
      return that.merge2(this);
    }
  }
  
  /** Merges two possibly null Faces. */
  public static Face merge(Face f1, Face f2) {
    return f1==null ? f2 : f1.merge(f2);
  }
  
  /** Removes 'this' from the head of its queue and returns the remaining queue. */
  public Face next() {
    Face ans = this.small==null ? this.big : this.small.merge(this.big);
    this.small = this.big = null; this.depth = 1;
    return ans;
  }
  
  /** Calls 'drawStrip()' for all Faces in the queue headed by this Face that overlap the strip of pixels with the specified x-coordinate, and returns a queue containing only those which extend further right.
   * @param u the x-coordinate of the strip of pixels.
   * @param zbuf an array with one entry for each pixel in the strip. The top pixel is at index '0'.
   * @param cbuf an array with one entry for each pixel in the strip.
   * @return the remaining queue of Faces, to be drawn on strips with larger x-coordinates.
   */
  public Face drawAll(int u, int[] zbuf, byte[] cbuf) {
    Face queue = this, ans = null;
    while (queue!=null && queue.l<=u) {
      Face f = queue; queue = queue.next();
      if (f.r>u) {
        f.draw(u, zbuf, cbuf);
        f.l = u + 1;
        ans = f.merge(ans);
      }
    }
    if (queue!=null) ans = queue.merge(ans);
    return ans;
  }
  
  /* Override things in Object. */
  
  public String toString() {
    return
      "Face[l="+this.l+", t="+this.t+", r="+this.r+", b="+this.b+
      ", colour="+this.colour+", dist="+this.dist+"]";
  }
  
  /* Protected. */
  
  /** Draws the part of this Face which overlaps the vertical strip of pixels with the specified x-coordinate. This Face must not be part of a queue when this method  is called.
   * <p>Subclasses should override this method to define the shape of the Face. A typical implementation would make one call to 'vline()'.
   * @param u the x-coordinate of the strip in pixels.
   * @param zbuf an array with one entry for each pixel in the strip. The top pixel is at index '0'. Each entry is a distance, in units such that the size of the landscape is '1&lt;&lt;16'.
   * @param cbuf an array with one entry for each pixel, giving its colour.
   */
  protected abstract void draw(int u, int[] zbuf, byte[] cbuf);
  
  /** Replaces the colour (but not the distance) of all pixels from 't' (inclusive) to 'b' (exclusive) with the colour of this Face, provided they are further away than this Face. This method assumes distance decreases from top to bottom. 't' and 'b' are clipped to the range '0' to 'zbuf.length'.
   * <p>If the colour of this Face is '0xff', then instead of replacing the colour of the pixels it simply clears the bottom three bits. This makes the triangle look like a shadow, with the original colour showing through but darkened.
   * <p>Subclasses can override this method to define a different plotting action.
   * @param t the y-coordinate of the first pixel of the vline.
   * @param b the y-coordinate of the first pixel below the vline.
   */
  protected void vline(int t, int b, int[] zbuf, byte[] cbuf) {
    if (t<0) t = 0;
    if (b>zbuf.length) b = zbuf.length;
    if (this.colour!=(byte)0xff) {
      while (t<b && zbuf[t]>this.dist) cbuf[t++] = this.colour;
    } else {
      while (t<b && zbuf[t]>this.dist) cbuf[t++] &= -8;
    }
  }
  
  /* Private. */
  
  /** Implements 'merge()' given that 'that' is non-null and comes after 'this'. */
  private Face merge2(Face that) {
    that = that.merge(this.small);
    if (this.big==null || that.depth>this.big.depth) {
      this.small = this.big; this.big = that;
    } else {
      this.small = that;
    }
    this.depth = this.small==null ? 1 : this.small.depth+1;
    return this;
  }
  
  private int l, t, r, b;
  private final int order, dist;
  private final byte colour;
  
  private Face big, small; // 'small.depth<big.depth'.
  private int depth; // 'small.depth+1'. 'null' considered to have depth '0'.
}
