package org.sc3d.apt.jrider.v1;

 /** The main game loop. */
public class Game extends Thread {
  /** Contructs a Game. The maximum number of players is defined by the length of 'sis' but players must be added to the game explicitly by calling 'setPlayerConfig()'.
   * @param controller a Controller that channels key presses to and from the server.
   * @param sis an array with one entry for each player. A non-null entry gives a SceneImage for displaying a player's view of the game. A null entry indicates that the player's view of the game should not be displayed, for example because they are on the other side of a network. Each SceneImage should be of the sort that draws itself in its own window. Note that sound is only generated for players with non-null SceneImages.
   * @param land the Landscape on which the race takes place. This also defines the course.
   * @param map the Map on which the Cars' positions should be shown, or 'null'. Obviously it should be a Map of 'land'.
   * @param inCar 'true' means a first-person viewpoint, 'false' means third-person.
   * @param fixedCamera If 'inCar', 'true' means a viewpoint that rotates with the Car, 'false' means a viewpoint that looks in the direction of motion. If '!inCar' then 'fixedCamera' is ignored.
   */
  public Game(
    Controller controller,
    SceneImage[] sis,
    Landscape land, Map map,
    boolean inCar, boolean fixedCamera
  ) {
    this.numCars = sis.length;
    this.controller = controller; this.sis = sis;
    this.land = land; this.map = map;
    this.inCar = inCar; this.fixedCamera = fixedCamera & inCar;
    this.configs = new PlayerConfig[this.numCars];
    this.setPriority(Thread.MIN_PRIORITY);
    this.shouldEnd = false;
  }

  /* New API. */
  
  /** Set this field to 'true' to tell this Game to end. If you need to block until it has ended, use 'join()'. */
  public boolean shouldEnd;
  
  /** Set the PlayerConfig for a player. The change takes effect next time 'escape' is pressed (or at the beginning of the game).
   * @param playerNumber the index of the player.
   * @param config the player's new configuration.
   */
  public void setPlayerConfig(int playerNumber, PlayerConfig config) {
    this.configs[playerNumber] = config;
  }
  
  /** Returns the configuration of the player (the one which will take effect the next time 'escape' is pressed). */
  public PlayerConfig getPlayerConfig(int playerNumber) {
    return this.configs[playerNumber];
  }
  
  /** Changes the SceneImage for player 'playerNumber' to be 'si'. This method must be called before 'start()'. */
  public void setSceneImage(int playerNumber, SceneImage si) {
    this.sis[playerNumber] = si;
  }
  
  /** Called each time a player legally crosses the start line. This version just prints statistics to 'System.out'.
   * @param lc the player's LapCounter.
   */
  public void crossLine(LapCounter lc) {
    System.out.println(lc.getLapCount()+" laps completed");
    System.out.println(lc.getLast()+"ms for last lap");
    System.out.println(lc.getBest()+"ms for best lap");
  }

  /* Override things in Thread. */
  
  /** The main loop of the Game. This method will return soon after 'shouldEnd' becomes 'true'. */
  public void run() {
    // Work out which players are local.
    int numLocals = 0;
    final int[] locals = new int[this.numCars];
    for (int i=0; i<this.numCars; i++)
      if (this.sis[i]!=null) locals[numLocals++] = i;
    // Initialise.
    final PlayerConfig[] pcs = new PlayerConfig[this.numCars];
    final LapCounter[] laps = new LapCounter[this.numCars];
    for (int i=0; i<this.numCars; i++)
      laps[i] = new LapCounter(this.land, i);
    final Car[] cars = new Car[this.numCars];
    final CollisionSphere[] css = new CollisionSphere[10*numCars];
    final FPSCounter fpsc = new FPSCounter();
    final EngineNoise en = new EngineNoise(numLocals);
    int mapCount = 0;
    // Loop once for each frame.
    while (!this.shouldEnd) for (int l=0; l<numLocals; l++) {
      // Render a frame for one local player.
      final int car = locals[l];
      if (pcs[car]!=null) en.setRevs(l, cars[car].getRevs());
      Camera c;
      if (pcs[car]==null) {
        c = laps[car].getStartLineCamera();
      } else if (this.fixedCamera) {
        c = cars[car].getFixedCamera();
      } else {
        c = cars[car].getCamera(this.inCar ? 0<<22 : 12<<22, land);
      }
      final Frame f = this.sis[car].reset(c, land);
      laps[car].addFacesTo(f);
      for (int i=0; i<this.numCars; i++) {
        if (pcs[i]!=null) cars[i].addFacesTo(f, this.inCar && i==car);
      }
      this.sis[car].doFrame();
      // Move the cursors on the Map.
      if ((mapCount++&0x3f)==0) {
        for (int i=0; i<numCars; i++) if (pcs[i]!=null) {
          final Trajectory t = cars[i].getTrajectory();
          map.setPos(i, t.x, t.y);
        }
        map.doFrame();
      }
      // Process keypresses until we catch up with real time.
      int[] keyData;
      do {
        this.yield();
        keyData = controller.getKeyData();
      } while (keyData.length==0);
      fpsc.doFrame(keyData.length, numLocals);
      for (int time=0; time<keyData.length; time++) {
        int spheresUsed = 0;
        for (int i=0; i<numCars; i++) if (pcs[i]!=null)
          spheresUsed = cars[i].getSpheres(css, spheresUsed);
        for (int i=0; i<spheresUsed; i++) {
          css[i].hitLand(this.land);
          for (int j=0; j<i; j++) css[i].hitSphere(css[j]);
        }
        for (int i=0; i<numCars; i++) if (pcs[i]!=null) {
          final int keys = 0x1f & (keyData[time] >> (5*i));
          cars[i].tick(keys, true);
          final Trajectory t = cars[i].getTrajectory();
          if (laps[i].tick(t)) this.crossLine(laps[i]);
          if (
            (keys==0x0b || laps[i].getDistance(t)>(12<<22)) &&
            t.vx<(1<<13) && t.vx>(-1<<13) &&
            t.vy<(1<<13) && t.vy>(-1<<13)
          ) { // Panic button.
            cars[i] = laps[i].getPanicPosition(pcs[i]);
          }
        }
        if ((keyData[time]&(1<<30)) != 0) { // Escape key pressed.
          System.arraycopy(this.configs, 0, pcs, 0, this.numCars);
          for (int i=0; i<this.numCars; i++) {
            laps[i] = new LapCounter(this.land, i);
            cars[i] =
              pcs[i]==null ? null : laps[i].getInitialPosition(pcs[i]);
          }
          for (int i=0; i<numLocals; i++) {
            en.setRevs(i, 0);
          }
        }
      }
    }
  }
  
  /* Private. */
  
  private final int numCars;
  private final Controller controller;
  private final SceneImage[] sis;
  private final PlayerConfig[] configs; // Read only when 'escape' is pressed.
  private final Landscape land;
  private final Map map;
  private final boolean inCar, fixedCamera;
  
  /* Test code. */
  public static void main(String[] args) {
    // Parse arguments.
    if (args.length!=4) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.Game <seed> <numCars> "+
      "<camera mode> <drive mode>"
    );
    final int numCars = Integer.parseInt(args[1]);
    String cameraMode = args[2].toUpperCase();
    int magnification = 2;
    if (cameraMode.endsWith("-BIG")) {
      magnification = 3;
      cameraMode = cameraMode.substring(0, cameraMode.length()-4);
    } else if (cameraMode.endsWith("-SMALL")) {
      magnification = 1;
      cameraMode = cameraMode.substring(0, cameraMode.length()-6);
    }
    final boolean fixedCamera = cameraMode.equals("FIXED");
    final boolean motionCamera = cameraMode.equals("MOTION");
    final boolean followCamera = cameraMode.equals("FOLLOW");
    if (!fixedCamera && !motionCamera && !followCamera)
      throw new IllegalArgumentException(
        "Camera mode may be 'fixed', 'motion' or 'follow', with an optional "+
        "suffix '-big' or '-small', but it may not be "+args[2]
      );
    final String driveMode = args[3].toUpperCase();
    final boolean frontWheelDrive = driveMode.equals("FRONT");
    final boolean backWheelDrive =
      driveMode.equals("BACK") || driveMode.equals("REAR");
    final boolean fourWheelDrive = driveMode.equals("FOUR");
    if (!frontWheelDrive && !backWheelDrive && !fourWheelDrive)
      throw new IllegalArgumentException(
        "Drive mode must be 'front', 'rear' or 'four', not "+args[3]
      );
    // Initialise.
    final java.awt.Frame pleaseWait = new java.awt.Frame("Please wait");
    pleaseWait.add(new java.awt.Label("Please wait"));
    pleaseWait.pack();
    pleaseWait.show();
    final Controller controller = new Controller(Controller.DEFAULT_KEYS);
    controller.printKeys(numCars);
    final int res = Math.max(128, 64*magnification);
    final Lens lens = new Lens(2*res, res, res, res);
    final SceneImage[] sis = new SceneImage[numCars];
    final PlayerConfig[] configs = new PlayerConfig[numCars];
    for (int car = 0; car<numCars; car++) {
      sis[car] = new SceneImage(
        lens,
        256*magnification, 128*magnification,
        Controller.PLAYERS[car]
      );
      sis[car].getCanvas().addKeyListener(controller);
    }
    final Landscape land = Landscape.generate(args[0], 11);
    final Map map = new Map(land, 8, "Map of '"+args[0]+"'");
    pleaseWait.hide();
    sis[0].getCanvas().requestFocus();
    final Game me = new Game(
      controller,
      sis,
      land, map,
      fixedCamera | motionCamera, fixedCamera
    );
    for (int i=0; i<numCars; i++) {
      me.setPlayerConfig(i, new PlayerConfig(
        frontWheelDrive | fourWheelDrive, backWheelDrive | fourWheelDrive,
        i, "Car "+i+" ("+Controller.PLAYERS[i]+")"
      ));
    }
    me.start();
  }
}
