package org.sc3d.apt.jrider.v1;

/** Represents the course that a player has to complete each lap, and provides related facilities, including a lap counter and stopwatch. */
public class LapCounter {
  /** Constructs a LapCounter given regularly-spaced Cambers all round the course.
   * @param land the Landscape on which the race will take place.
   * @param playerNumber the player number from '0' to '5'.
   */
  public LapCounter(final Landscape land, int playerNumber) {
    this.cs = land.getCambers(); this.pNum = playerNumber;
    this.lapCount = -1; this.count = cs.length-4;
    this.time = -1; this.last = -1; this.best = -1;
  }
  
  /* New API. */
  
  /** Returns a Car on the starting grid. Cars '2n' and '2n+1' start side-by-side on the Camber with index 'length-n-1'.
   * @param config a PlayerConfig giving the properties of the player's car.
   */
  public Car getInitialPosition(PlayerConfig config) {
    final Camber c = this.cs[this.cs.length + (~this.pNum>>1)];
    final int side = ((this.pNum&1)==0) ? -96 : 96;
    return this.getPosition(
      c.x + side*(c.xx>>8),
      c.y + side*(c.xy>>8),
      c.z + side*(c.xz>>8) + (1<<22),
      -c.xy, c.xx,
      config
    );
  }
  
  /** Returns a Car in a place from which it can plausibly continue the race.
   * @param config a PlayerConfig giving the properties of the player's car.
   */
  public Car getPanicPosition(PlayerConfig config) {
    final Camber c = this.cs[this.count];
    return this.getPosition(c.x, c.y, c.z+(1<<22), -c.xy, c.xx, config);
  }
  
  /** Returns a Camera that shows the start line from the left-hand side. */
  public Camera getStartLineCamera() {
    final Camber c = this.cs[0];
    int l = (Math.max(Math.abs(c.xx), Math.abs(c.xy)) >> 14) + 1;
    int wx = c.xx/l, wy = c.xy/l;
    for (int i=0; i<5; i++) {
      l = (3<<14) - ((wx*wx+wy*wy) >> 14);
      wx = (l*wx) >> 15; wy = (l*wy) >> 15;
    }
    return new Camera(
      c.x - c.xx - wx*(2<<22-14),
      c.y - c.xy - wy*(2<<22-14),
      c.z - c.xz + (1<<22),
      wx<<2, wy<<2
    );
  }
  
  /** Renders the next few Cambers as lines painted on the road, and adds the resulting queue of Faces to 'f'. */
  public void addFacesTo(Frame f) {
    final Camera c = f.camera;
    final Lens l = f.lens;
    final int centreu = l.width << 7, centrev = l.height << 7;
    final int lensu = l.lensu << 8, lensv = l.lensv << 8;
    final int[] xs = new int[4], ys = new int[4], zs = new int[4];
    final int[] us = new int[4], vs = new int[4], ws = new int[4];
    Camber prev = this.cs[this.count];
    for (int i=1; i<=20; i++) {
      // Work out the coordinates of the corners.
      final Camber ci = this.cs[(this.count+i)%this.cs.length];
      final int x1 = ci.x - 3*((ci.x-prev.x)>>2);
      final int y1 = ci.y - 3*((ci.y-prev.y)>>2);
      final int z1 = ci.z - 3*((ci.z-prev.z)>>2);
      final int x3 = ci.x - 1*((ci.x-prev.x)>>2);
      final int y3 = ci.y - 1*((ci.y-prev.y)>>2);
      final int z3 = ci.z - 1*((ci.z-prev.z)>>2);
      final int xx1 = ci.xx - 3*((ci.xx-prev.xx)>>2);
      final int xy1 = ci.xy - 3*((ci.xy-prev.xy)>>2);
      final int xz1 = ci.xz - 3*((ci.xz-prev.xz)>>2);
      final int xx3 = ci.xx - 1*((ci.xx-prev.xx)>>2);
      final int xy3 = ci.xy - 1*((ci.xy-prev.xy)>>2);
      final int xz3 = ci.xz - 1*((ci.xz-prev.xz)>>2);
      xs[0] = x1 - (xx1>>5); ys[0] = y1 - (xy1>>5); zs[0] = z1 - (xz1>>5);
      xs[1] = x1 + (xx1>>5); ys[1] = y1 + (xy1>>5); zs[1] = z1 + (xz1>>5);
      xs[2] = x3 - (xx3>>5); ys[2] = y3 - (xy3>>5); zs[2] = z3 - (xz3>>5);
      xs[3] = x3 + (xx3>>5); ys[3] = y3 + (xy3>>5); zs[3] = z3 + (xz3>>5);
      prev = ci;
      // Rotate them into the viewer's frame.
      for (int j=0; j<4; j++) {
        final long x = (c.wy*(long)(xs[j]-c.x) - c.wx*(long)(ys[j]-c.y)) >> 16;
        final long y = (c.wx*(long)(xs[j]-c.x) + c.wy*(long)(ys[j]-c.y)) >> 16;
        final long z = zs[j]-c.z;
        if (y>0) {
          us[j] = centreu + (int)((x*lensu) / y);
          vs[j] = centrev - (int)((z*lensv) / y);
        }
        ws[j] = (int)(y >> 16);
      }
      // Construct the Faces.
      Face ans = Triangle.make(
        us[2], vs[2], ws[2],
        us[1], vs[1], ws[1],
        us[0], vs[0], ws[0],
        Face.ORDER_PAINT,
        (byte)0x2F,
        l.width, l.height
      );
      if (ans!=null) f.faces = ans.merge(f.faces);
      ans = Triangle.make(
        us[1], vs[1], ws[1],
        us[2], vs[2], ws[2],
        us[3], vs[3], ws[3],
        Face.ORDER_PAINT,
        (byte)0x2F,
        l.width, l.height
      );
      if (ans!=null) f.faces = ans.merge(f.faces);
    }
  }
  
  /** Call this exactly once every millisecond to inform the LapCounter of the position of the Car.
   * @param t the Trajectory of the Car (the velocity is ignored).
   * @return 'true' once when the Car completes a Lap, otherwise 'false'.
   */
  public boolean tick(Trajectory t) {
    if (this.time>=0) this.time++;
    final int next = (this.count+1) % this.cs.length;
    final Camber c = this.cs[next];
    if (((t.y-c.y)>>16)*(c.xx>>16)-((t.x-c.x)>>16)*(c.xy>>16)<0) return false;
    // Crossed a Camber.
    this.count = next;
    if (next!=0) return false;
    // Crossed the start line.
    this.lapCount++; this.last = this.time; this.time = 0;
    if (this.best==-1 || this.last<this.best) this.best = this.last;
    return true;
  }
  
  /** Returns the approximate distance from the Car to the last Camber. This is used to decide whether to panic and reset the position of the Car.
   * @param t the Trajectory of the Car (the velocity is ignored).
   * @return the approximate distance, in units such that the size of the landscape is '1&lt;&lt;32'. The distance may be an underestimate by up to a factor of 'sqrt(2)'.
   */
  public int getDistance(Trajectory t) {
    final Camber c = this.cs[this.count];
    return Math.max(Math.abs(t.x-c.x), Math.abs(t.y-c.y));
  }
  
  /** Returns the number of completed laps, or '-1' if the player has not yet crossed the start line. */
  public int getLapCount() { return this.lapCount; }
  
  /** Returns the index of the most recently passed Camber. */
  public int getCount() { return this.count; }
  
  /** Returns the number of milliseconds that have elapsed since the player crossed the start line, or '-1' if the player has not yet crossed the start line. */
  public int getTime() { return this.time; }
  
  /** Returns the time in milliseconds of the most recently completed lap, or '-1' if the player has not yet completed a lap. */
  public int getLast() { return this.last; }
  
  /** Returns the best lap time so far in milliseconds, or '-1' if the player has not yet completed a lap. */
  public int getBest() { return this.best; }
  
  /* Private. */
  
  /** Returns a Car at 'x, y, z' facing 'yx, yy, 0'. */
  private Car getPosition(
    int x, int y, int z,
    int yx, int yy,
    PlayerConfig config
  ) {
    int l = (Math.max(Math.abs(yx), Math.abs(yy)) >> 14) + 1;
    yx /= l; yy /= l;
    for (int i=0; i<5; i++) {
      l = (3<<14) - ((yx*yx + yy*yy) >> 14);
      yx = (l*yx) >> 15; yy = (l*yy) >> 15;
    }
    return new Car(
      new Trajectory(x, y, z),
      new Orientation(yy, -yx, 0, yx, yy, 0, 0, 0, 1<<14),
      config
    );
  }

  private final Camber[] cs;
  private final int pNum;
  private int lapCount, count;
  private int time, last, best;
}
