package org.sc3d.apt.jrider.v1;

import java.awt.image.*;

/** A subclass of BufferedImage for displaying an animation of a rendered Landscape with shadows and objects superimposed using Faces. The scene is rendered in vertical strips. The BufferedImage is of type 'TYPE_BYTE_INDEXED' and the ColorModel used is 'LandGen.ICM'.
 * <p>To draw a frame, make the following sequence of calls:<ul>
 * <li>Call 'reset()', passing the Landscape (if any) and the position of the Camera. This returns a Frame 'f'.
 * <li>Merge Faces into 'f.objects' and 'f.shadows'. Most objects that generate Faces provide some sort of 'addFacesTo()' method to help with this.
 * <li>Call 'doFrame()'.
 * <li>Draw this Image on the screen.
 * </ul>
 * If you want the Image to be drawn only on one particular Canvas, this class can do the fourth step for you.
 */
public class SceneImage extends BufferedImage {
  /** Constructs a SceneImage.
   */
  public SceneImage(Lens lens) {
    super(
      lens.width, lens.height,
      BufferedImage.TYPE_BYTE_INDEXED,
      LandGen.ICM
    );
    this.lens = lens;
    this.frame = null;
  }
  
  /** Constructs a SceneImage that is displayed in its own window. The Image will be drawn on the screen automatically each time 'doFrame()' is drawn.
   * @param lens the Lens to use.
   * @param width the width of the display. If this is larger than 'lens.width' then the Image will be scaled.
   * @param height the height of the display. If this is larger than 'lens.height' then the Image will be scaled.
   * @param title the title of the window.
   */
  public SceneImage(
    Lens lens,
    int width, int height,
    String title
  ) {
    this(lens);
    this.setCanvas(new ImageCanvas(width, height, this, title));
  }
  
  /* New API. */
  
  /** Returns the Canvas whose 'doFrame()' method is called each time 'doFrame()' is called.
   * @return the Canvas, or 'null'.
   */
  public ImageCanvas getCanvas() { return this.canvas; }
  
  /** Sets the Canvas on whose 'doFrame()' method is called each time 'doFrame()' is called. This SceneImage will be scaled to cover the ImageCanvas.
   * @param canvas an ImageCanvas, or 'null' to disable automatic drawing.
   */
  public void setCanvas(ImageCanvas canvas) { this.canvas = canvas; }
  
  /** Starts a new frame. All information concerning the previous frame is discarded.
   * @param camera the position and orientation of the Camera.
   * @param land the Landscape to render.
   * @return a Frame to which objects and shadows can be added. Note that its 'lens' field may not be the lens passed to the constructor; it may describe a larger rectangle if the Camera is not upright, in order to contain the whole field of view.
   */
  public Frame reset(final Camera camera, final Landscape land) {
    // Work out the size of the rectangle that needs to be drawn.
    final int W = this.lens.width, H = this.lens.height;
    final int LU = this.lens.lensu, LV = this.lens.lensv;
    final int uu = camera.cos, uv = camera.sin * LV / LU;
    final int vu = -camera.sin * LU / LV, vv = camera.cos;
    final int RW = 1 + (((W-1)*Math.abs(uu) + (H-1)*Math.abs(vu)) >> 16);
    final int RH = 1 + (((W-1)*Math.abs(uv) + (H-1)*Math.abs(vv)) >> 16);
    // Construct and return the Frame.
    this.frame = new Frame(land, camera, new Lens(RW, RH, LU, LV));
    return this.frame;
  }
  
  /** Draws the current Frame on this Image. The Landscape is drawn first, then the shadows are added, and finally the objects. The Frame becomes invalid. To draw another Frame you have to call 'reset()' again. */
  public synchronized void doFrame() {
    // Make convenient names for properties of 'this.frame'.
    // The initial 'R' indicates the rotated coordinate system.
    final Camera c = this.frame.camera;
    final int RW = this.frame.lens.width, RH = this.frame.lens.height;
    final int RLU = this.frame.lens.lensu, RLV = this.frame.lens.lensv;
    // Render the scene.
    final int detail = RLU/4;
    if (this.pixels==null || RH*RW>this.pixels.length)
      this.pixels = new byte[RH*RW]; // Note transposed.
    final int[] ZBUF = new int[RH];
    final byte[] CBUF = new byte[RH];
    int zdv = (1<<16)/RLV;
    int zdw = -zdv * (RH-1) / 2;
    // Do each strip in turn, from left to right.
    for (int u=0; u<RW; u++) {
      if (this.frame.land!=null) {
        int xdw = c.wx + c.wy * (2*u-RW+1) / (2*RLU);
        int ydw = c.wy - c.wx * (2*u-RW+1) / (2*RLU);
        this.frame.land.render(
          c.x, c.y, c.z, xdw, ydw, zdw, zdv, ZBUF, CBUF, detail, 0//seed
        );
        seed *= 487654853;
      } else {
        for (int v=0; v<RH; v++) {
          ZBUF[v] = 1<<16; CBUF[v] = (byte)0xff;
        }
      }
      if (this.frame.faces!=null)
        this.frame.faces = this.frame.faces.drawAll(u, ZBUF, CBUF);
      // Copy strip to image.
      System.arraycopy(CBUF, 0, this.pixels, RH*u, RH);
    }
    // Perform the rotation.
    final int W2 = this.lens.width, H2 = this.lens.height;
    final int LU2 = this.lens.lensu, LV2 = this.lens.lensv;
    final int uu = c.cos * RLU / LU2, uv = c.sin * RLV / LU2;
    final int vu = -c.sin * RLU / LV2, vv = c.cos * RLV / LV2;
    final byte[] PIX2 =
      ((DataBufferByte)this.getRaster().getDataBuffer()).getData(); // 'W2*H2'.
    int i = 0;
    for (int v2=0; v2<H2; v2++) {
      int u = ((RW<<16) - (W2-1)*uu + (2*v2+1-H2)*vu) >> 1;
      int v = ((RH<<16) - (W2-1)*uv + (2*v2+1-H2)*vv) >> 1;
      for (int u2=0; u2<W2; u2++) {
        PIX2[i++] = this.pixels[(v>>16) + RH*(u>>16)];
        u += uu; v += uv;
      }
    }
    // Finish the frame.
    this.frame = null;
    if (this.canvas!=null) this.canvas.doFrame();
  }
  
  /* Private. */

  private Lens lens; // The Lens passed to the constructor.
  private Frame frame; // The current Frame.
  private byte[] pixels = null; // An array used by 'doFrame()'.
  private ImageCanvas canvas = null; // The canvas on which to draw, or 'null'.
  
  private static int seed = 1;
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.SceneProducer <seed>"
    );
    final Lens lens = new Lens(256, 128, 128, 128);
    SceneImage me = new SceneImage(lens, 512, 256, "Testing SceneImage");
    Landscape land = Landscape.generate(args[0], 11);
    int cx = 0, cy = 0;
    int wx = (3<<16)/5, wy = (4<<16)/5;
    long time = System.currentTimeMillis();
    int count = 0;
    while (true) {
      int cz = land.getHeight(cx, cy) + (1<<24);
      Frame f = me.reset(new Camera(cx, cy, cz, wx, wy), land);
      f.faces = Triangle.make(
        200<<8, 100<<8, 1<<13,
        300<<8, 300<<8, 1<<12,
        100<<8, 200<<8, 1<<11,
        Face.ORDER_OBJECT,
        (byte)0x87,
        lens.width, lens.height
      );
      me.doFrame();
      cx += wx<<6; cy += wy<<6;
      if (++count>=100) {
        long newTime = System.currentTimeMillis();
        System.out.println("fps: "+(100000.0/(newTime-time)));
        time = newTime; count = 0;
      }
    }
  }
}
