package org.sc3d.apt.jrider.v1;

/** A subclass of Face that represents a flat triangle. */
public class Triangle extends Face {
  /** The public way to construct Triangles is using the 'make()' method. */
  private Triangle(
    int l, int t, int r, int b,
    int order, int dist,
    byte colour
  ) {
    super(l, t, r, b, order, dist, colour);
  }
  
  /* New API. */
  
  /** Constructs a Triangle given its vertices. The vertices must be presented in clockwise order as seen from the visible side of the Triangle. 'null' is returned if any of the following is true:<ul>
   * <li>At least one vertex has a negative distance.
   * <li>At least one coordinate has a magnitude exceeding about '1&lt;&lt;20'.
   * <li>The Triangle is obviously off the screen.
   * </ul>
   * @param u1 the screen x-coordinate of the first vertex, in units such that the width of a pixel is '1&lt;&lt;8'.
   * @param v1 the screen y-coordinate of the first vertex, in units such that the hieght of a pixel is '1&lt;&lt;8'.
   * @param w1 the distance to the first pixel, in units such that the size of the landscape is '1&lt;&lt;16'.
   * @param u2 the x-coordinate of the second vertex.
   * @param v2 the y-coordinate of the second vertex.
   * @param w2 the distance to the second vertex.
   * @param u3 the x-coordinate of the third vertex.
   * @param v3 the y-coordinate of the third vertex.
   * @param w3 the distance to the third vertex.
   * @param order the plot order of the triangle.
   * @param colour the colour of the triangle.
   * @param width the width of the screen in pixels.
   * @param height the height of the screen in pixels.
   * @return the new Triangle, or 'null' if nothing is visible.
   */
  public static Triangle make(
    int u1, int v1, int w1,
    int u2, int v2, int w2,
    int u3, int v3, int w3,
    int order,
    byte colour,
    int width, int height
  ) {
    int l = u1; if (u2<l) l = u2; if (u3<l) l = u3; l = (l+0x80) >> 8;
    if (l>=width || l<(-1<<12)) return null;
    int t = v1; if (v2<t) t = v2; if (v3<t) t = v3; t = (t+0x80) >> 8;
    if (t>=height || t<(-1<<12)) return null;
    int r = u1; if (u2>r) r = u2; if (u3>r) r = u3; r = (r+0x80) >> 8;
    if (r<0 || r>(1<<12)) return null;
    int b = v1; if (v2>b) b = v2; if (v3>b) b = v3; b = (b+0x80) >> 8;
    if (b<0 || b>(1<<12)) return null;
    int dist = w1; if (w2<dist) dist = w2; if (w3<dist) dist = w3;
    if (dist<0) return null;
    Triangle ans = new Triangle(l, t, r, b, order, dist, colour);
    ans.e1du = v2 - v3; ans.e1dv = u3 - u2; ans.e1 =
      (((0x80 - (u2&0xff)) * ans.e1du) >> 8) - ((u2>>8) * ans.e1du) +
      (((-0x80 - (v2&0xff)) * ans.e1dv) >> 8) - ((v2>>8) * ans.e1dv);
    if (
      ((((u1-u2)&0xff) * ans.e1du) >> 8) + (((u1-u2)>>8) * ans.e1du) +
      ((((v1-v2)&0xff) * ans.e1dv) >> 8) + (((v1-v2)>>8) * ans.e1dv) < 0
    ) return null;
    ans.e2du = v3 - v1; ans.e2dv = u1 - u3; ans.e2 =
      (((0x80 - (u3&0xff)) * ans.e2du) >> 8) - ((u3>>8) * ans.e2du) +
      (((-0x80 - (v3&0xff)) * ans.e2dv) >> 8) - ((v3>>8) * ans.e2dv);
    ans.e3du = v1 - v2; ans.e3dv = u2 - u1; ans.e3 =
      (((0x80 - (u1&0xff)) * ans.e3du) >> 8) - ((u1>>8) * ans.e3du) +
      (((-0x80 - (v1&0xff)) * ans.e3dv) >> 8) - ((v1>>8) * ans.e3dv);
    return ans;
  }
  
  /* Override things in Face. */
  
  /** Draws that part of this Triangle that overlaps the strip of pixels with the specified x-coordinate. */
  protected void draw(int u, int[] zbuf, byte[] cbuf) {
    int t = 0, b = zbuf.length, v;
    if (this.e1dv!=0) {
      v = ~(this.e1 + u*this.e1du) / this.e1dv;
      if (this.e1dv>0 && v>t) t = v;
      if (this.e1dv<0 && v<b) b = v;
    }
    if (this.e2dv!=0) {
      v = ~(this.e2 + u*this.e2du) / this.e2dv;
      if (this.e2dv>0 && v>t) t = v;
      if (this.e2dv<0 && v<b) b = v;
    }
    if (this.e3dv!=0) {
      v = ~(this.e3 + u*this.e3du) / this.e3dv;
      if (this.e3dv>0 && v>t) t = v;
      if (this.e3dv<0 && v<b) b = v;
    }
    this.vline(t, b, zbuf, cbuf);
  }
  
  /* Private. */
  
  private int e1, e1du, e1dv;
  private int e2, e2du, e2dv;
  private int e3, e3du, e3dv;
}
