package org.sc3d.apt.jrider.v1;

/** Represents part of an object for the purposes of collision detection. */
public class CollisionSphere {
  /** Constructs a CollisionSphere, given its position, radius, velocity and angular velocity, its material properties, and the Body of which it is part.
   * @param t the Trajectory of the centre of the sphere.
   * @param o an Orientation from which to extract the angular velocity of the sphere.
   * @param radius the radius of the sphere in millimetres, so that the size of the landscape is '1&lt;&lt;20'. The radius should not be more than 64m ('1&lt;&lt;16').
   * @param stiffness the contact force that the sphere will generate if deformed, in units of 1MN/m.
   * @param damping the normal damping coefficient, expressed as a time in units of '1ms'. The damping force for a collision at a given velocity is equal to the extra contact force that would be experienced if the deformation were allowed to grow unchecked for this amount of time. For critical damping use about 'sqrt(mass/stiffness/2)'.
   * @param mu the coefficient of tangential friction, in units such that '1&lt;&lt;8' means that the maximum frictional force is equal to the contact force.
   * @param mass an underestimate of the mass of the RigidBody. This is used to calculate the tangential frictional force when the sliding speed is low. The law used is that the force will be strong enough to bring a body of the specified mass to rest in 1ms. Too low, and the body will slide slowly downhill instead of stopping. Too high and the friction will over-compensate, potentially unstably.
   * @param body the RigidBody.
   */
  public CollisionSphere(
    Trajectory t, Orientation o,
    int radius,
    int stiffness, int damping,
    int mu, int mass,
    Body body
  ) {
    this.x = t.x; this.y = t.y; this.z = t.z;
    this.radius = radius << 12;
    this.vx = t.vx; this.vy = t.vy; this.vz = t.vz;
    this.wx = o.wx; this.wy = o.wy; this.wz = o.wz;
    this.stiffness = stiffness; this.damping = damping;
    this.mu = mu; this.mass = mass;
    this.body = body;
  }
  
  /* New API. */
  
  /** Applies a force to 'body' if this CollisionSphere overlaps with the specified Landscape.
   * <p>The approximation used is as follows. The height of the Landscape is sampled at four points around the shadow of the sphere, and a plane is constructed in such a way as to minimise the sum of the squared distances of the points from the plane. The sphere is then compared with the plane.
   * @param land the Landscape.
   * @return 'true' if this CollisionSphere skids on the Landscape enough to make a mark, 'false' if it rolls or does not touch it at all.
   */
  public boolean hitLand(Landscape land) {
    int h1 = this.z - land.getHeight(this.x-this.radius, this.y);
    int h2 = this.z - land.getHeight(this.x+this.radius, this.y);
    int h3 = this.z - land.getHeight(this.x, this.y-this.radius);
    int h4 = this.z - land.getHeight(this.x, this.y+this.radius);
    if (
      h1>=this.radius && h2>=this.radius &&
      h3>=this.radius && h4>=this.radius
    ) return false;
    // Work out the normal vector, pointing down, of length '1<<8'.
    int nx = (h1 - h2) / (this.radius >> 8);
    int ny = (h3 - h4) / (this.radius >> 8);
    int nz = (-1<<16) / this.length(nx, ny, -1<<8);
    nx = (0x80 - nx*nz) >> 8; ny = (0x80 - ny*nz) >> 8;
    // Work out the distance of the centre from the plane.
    int n = -((h1 + h2 + h3 + h4) >> 10) * nz;
    if (n>=this.radius) return false;
    // Apply appropriate forces.
    final boolean skid = this.hit(null, this.radius-n, nx, ny, nz);
    if (skid) land.drawSkidMark(this.x, this.y);
    return skid;
  }
  
  /** Applies equal and opposite forces to 'this.body' and 'that.body' if 'this' overlaps with 'that'.
   * @param that another CollisionSphere.
   * @return 'true' if 'this' and 'that' touch and skid enough to make a mark, 'false' if they roll or don't touch at all.
   */
  public boolean hitSphere(CollisionSphere that) {
    if (this.body==that.body) return false;
    // Work out the distance between the centres.
    int rr = this.radius + that.radius;
    int nx = that.x - this.x; if (nx>=rr || nx<=-rr) return false;
    int ny = that.y - this.y; if (ny>=rr || ny<=-rr) return false;
    int nz = that.z - this.z; if (nz>=rr || nz<=-rr) return false;
    int n = this.length(nx, ny, nz); if (n>=rr) return false;
    if (n==0) n = 1; // This is a hack and Oggie is ashamed.
    // Apply appropriate forces.
    return this.hit(that, rr-n, (nx<<8)/n, (ny<<8)/n, (nz<<8)/n);
  }
  
  /* Inner classes. */
  
  public static abstract class Body {
    /** Applies the specified force to this Body at the specified point.
     * @param fx the x-component of the force, in units such that 1N is '1&lt;&lt;2'.
     * @param fy the y-component of the force.
     * @param fz the z-component of the force.
     * @param x the x-coordinate of the point of application, in units such that the size of the landscape is '1&lt;&lt;32'.
     * @param y the y-coordinate of the point.
     * @param z the z-coordinate of the point.
     */
    public abstract void applyForce(
      int fx, int fy, int fz,
      int x, int y, int z
    );
  }
  
  /* Private. */
  
  /** Applies contact, damping and frictional forces appropriate for a collision at the specified point with the specified relative velocity. The point must be inside this CollisionSphere.
   * @param that the other CollisionSphere, or 'null' for an infinite mass object such as the Landscape.
   * @param deformation the total deformation of the two objects, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param nx the x-component of a vector of length '1&lt;&lt;8' in the direction from the centre of this sphere to the collision point.
   * @param ny the y-component of the normal vector.
   * @param nz the z-component of the normal vector.
   * @return 'true' if the objects skid enough to make a mark, 'false' if they roll.
   */
  private boolean hit(
    final CollisionSphere that,
    final int deformation,
    final int nx, int ny, int nz
  ) {
    // Share the deformation between the spheres.
    final int d1;
    if (that==null) d1 = deformation; else {
      int f = (that.stiffness<<8) / (this.stiffness+that.stiffness);
      d1 = f * (deformation>>8);
    }
    final int d2 = deformation - d1;
    // Work out the sum of (distance * angular velocity) for the two spheres
    // in units such that 1m/s is '1<<16'.
    int n = (this.radius - d1) >> 14;
    int nwx = n * this.wx, nwy = n * this.wy, nwz = n * this.wz;
    if (that!=null) {
      n = (that.radius - d2) >> 14;
      nwx += n * that.wx; nwy += n * that.wy; nwz += n * that.wz;
    }
    // Work out the passing velocity (this relative to that).
    int px = this.vx + ((nwy*nz - nwz*ny) >> 12);
    int py = this.vy + ((nwz*nx - nwx*nz) >> 12);
    int pz = this.vz + ((nwx*ny - nwy*nx) >> 12);
    if (that!=null) { px -= that.vx; py -= that.vy; px -= that.vz; }
    // Separate the velocity into normal and tangential components.
    final int pn = (px*nx + py*ny + pz*nz) >> 8;
    final int ptx = px - ((pn*nx) >> 8);
    final int pty = py - ((pn*ny) >> 8);
    final int ptz = pz - ((pn*nz) >> 8);
    final int pt = this.length(ptx, pty, ptz);
    // Work out the normal force.
    int fn = (d1 + this.damping*pn) * this.stiffness;
    if (that!=null) fn += that.damping*pn * that.stiffness;
    if (fn<0) fn = 0; else if (fn>(1<<22)) fn = 1<<22;
    final int fnx = (fn*nx) >> 8, fny = (fn*ny) >> 8, fnz = (fn*nz) >> 8;
    // Work out the tangential force.
    int mu = this.mu; if (that!=null && that.mu<mu) mu = that.mu; // Maybe max?
    int mass = this.mass; if (that!=null && that.mass<mass) mass = that.mass;
    final int ratio = mass*pt > (mu*fn)>>8 ? mu*fn/pt : mass<<8;
    final int ftx = (ptx * ratio) >> 8;
    final int fty = (pty * ratio) >> 8;
    final int ftz = (ptz * ratio) >> 8;
    // Apply the forces.
    n = (this.radius - d1)>>8;
    final int cx = this.x + n*nx, cy = this.y + n*ny, cz = this.z + n*nz;
    final int fx = fnx + ftx, fy = fny + fty, fz = fnz + ftz;
    this.body.applyForce(-fx, -fy, -fz, cx, cy, cz);
    if (that!=null) that.body.applyForce(fx, fy, fz, cx, cy, cz);
    return mass*pt > (mu*fn)>>7 &&
      (ftx>1<<13 || ftx<-1<<13 || fty>1<<13 || fty<-1<<13);
  }
  
  /** Returns the length of the specified vector, with less than 1% error. The vector must have a length of at least '1<<8'. */
  private int length(int x, int y, int z) {
    int size = (x<0 ? ~x : x) | (y<0 ? ~y : y) | (z<0 ? ~z : z), shift = 0;
    if (size>=(1<<23)) { size >>= 16; shift += 16; }
    if (size>=(1<<15)) { size >>= 8; shift += 8; }
    if (size>=(1<<11)) { size >>= 4; shift += 4; }
    if (size>=(1<<9)) { size >>= 2; shift += 2; }
    if (size>=(1<<8)) { size >>= 1; shift += 1; }
    x >>= shift; y >>= shift; z >>= shift;
    size = x*x + y*y + z*z; if (size>=(1<<16)) { size >>= 2; shift += 1; }
    return this.L[size>>8] << shift;
  }
  
  /** A look-up table for 'length()'. 'L[i]' is the square root of 'i<<8' rounded to the nearest integer. */
  private static final short[] L = new short[0x100];
  static {
    int n = 0;
    for (short i=0; i<0x100; i++) {
      int i2 = (i*i + i) >> 8;
      while (n<=i2) L[n++] = i;
    }
  }
  
  private int x, y, z;
  private int radius;
  private int vx, vy, vz;
  private int wx, wy, wz;
  private int stiffness, damping;
  private int mu, mass;
  private Body body;
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax: java org.sc3d.apt.jrider.v1.CollisionSphere <seed>"
    );
    final Landscape land = Landscape.generate(args[0], 11);
    final RigidBody wheel =
      new RigidBody(50, 2, 2, 2, 0, 0, land.getHeight(0, 0)+(1<<24));
    wheel.accelerate(0, 0, -1<<12);
    final Lens lens = new Lens(256, 256, 128, 256);
    final SceneImage si =
      new SceneImage(lens, 512, 256, "Testing CollisionSphere");
    final int[] x = new int[] {100}, y = new int[] {0}, z = new int[] {0};
    final int[] radii = new int[] {500};
    final int stiffness = 1, damping = 2;
    final int mu = 2<<8, mass = 2;
    // Animation.
    long time = System.currentTimeMillis();
    while (true) {
      final Trajectory ct = wheel.getTrajectory();
      final int wx = (-3<<16)/5, wy = (4<<16)/5;
      final int cx = ct.x - (wx<<9), cy = ct.y - (wy<<9);
      final int cz = Math.max(ct.z + (1<<23),land.getHeight(cx, cy) + (1<<22));
      final Camera c = new Camera(cx, cy, cz, wx, wy);
      Frame f = si.reset(c, land);
      Model.WHEEL.project(wheel.getTrajectory(), wheel.getOrientation(), f);
      si.doFrame();
      long nextTime = System.currentTimeMillis();
      while (time<nextTime) {
        wheel.accelerate(0, 0, -10<<2);
        final Orientation o = wheel.getOrientation();
        final Trajectory t = o.translate(wheel.getTrajectory(), 100, 0, 0);
        CollisionSphere me =
          new CollisionSphere(t, o, 500, stiffness, damping, mu, mass, wheel);
        me.hitLand(land);
        wheel.tick();
        time++;
      }
    }
  }
}
