package org.sc3d.apt.jrider.v1;

import java.awt.image.*;

/** Represents a window that displays a map of a Landscape, and allows coloured cursors to move around it.
 * <p><em>The API currently provided here is really inflexible. It would be better for this class to implement ImageProducer, or something. I'll redesign it later.</em>
 */
public class Map {
  /** Constructs and displays a Map.
   * @param land the Landscape to display.
   * @param log2size the logarithm to base 2 of the size of the Map in pixels.
   * @param title a string to use in the title bar of the Map.
   */
  public Map(Landscape land, int log2size, String title) {
    this.land = land; this.log2size = log2size;
    final int shift = 32-log2size, size = 1<<log2size;
    final BufferedImage image = new BufferedImage(
      size, size,
      BufferedImage.TYPE_BYTE_INDEXED,
      LandGen.ICM
    );
    this.pixels = ((DataBufferByte)image.getRaster().getDataBuffer()).getData();
    this.drawLand(0, 0, size, size);
    this.canvas = new ImageCanvas(size, size, image, title);
  }
  
  /* New API. */
  
  /** Sets the position of one of the cursors on the Map. Does not update the screen.
   * @param team a number from '0' to '5' indicating which cursor to move.
   * @param x the new x-coordinate (east) of the cursor, in units such that the size of the landscape is '1&lt;&lt;32'.
   * @param y the new y-coordinate (north) of the cursor.
   */
  public void setPos(int team, int x, int y) {
    final int shift = 32-this.log2size;
    this.drawLand(
      this.lastX[team]-1, this.lastY[team]-1,
      3, 3
    );
    this.lastX[team] = x >>> shift;
    this.lastY[team] = ~(y >>> shift);
    this.drawRect(
      this.lastX[team]-1, this.lastY[team]-1,
      3, 3,
      (byte)(0x8f+8*team)
    );
  }
  
  /** Updates the screen. */
  public void doFrame() { this.canvas.doFrame(); }
  
  /* Private. */
  
  /** Copied a rectangle of colours from 'land' to 'pixels'. If the specified coordinates are out of range, they will be wrapped correctly. Does not call 'newPixels()'.
   * @param x the left coordinate in pixels.
   * @param y the top coordinate in pixels.
   * @param w the width in pixels.
   * @param h the height in pixels.
   */
  private void drawLand(final int x, final int y, final int w, final int h) {
    final int m = (1<<this.log2size)-1, shift = 32-this.log2size;
    for (int py=y; py<y+h; py++) for (int px=x; px<x+h; px++) {
      this.pixels[(px&m)+((py&m)<<this.log2size)] =
        this.land.getColour(px<<shift, (~py)<<shift, this.log2size);
    }
  }
  
  /** Draws a solid rectangle on 'pixels'. If the specified coordinates are out of range, they will be wrapped correctly. Does not call 'newPixels()'.
   * @param x the left coordinate in pixels.
   * @param y the top coordinate in pixels.
   * @param w the width in pixels.
   * @param h the height in pixels.
   * @param colour a colour number interpreted using 'LandGen.ICM'.
   */
  private void drawRect(
    final int x, final int y,
    final int w, final int h,
    final byte colour
  ) {
    final int m = (1<<this.log2size)-1;
    for (int py=y; py<y+h; py++) for (int px=x; px<x+h; px++) {
      this.pixels[(px&m)+((py&m)<<this.log2size)] = colour;
    }
  }
  
  private Landscape land;
  private int log2size;
  private byte[] pixels;
  private ImageCanvas canvas;
  
  private final int[] lastX = new int[6], lastY = new int[6];
  
  /* Test code. */
  
  public static void main(String[] args) {
    if (args.length!=1) throw new IllegalArgumentException(
      "Syntax java org.sc3d.apt.jrider.v1.Map <seed>"
    );
    final Landscape land = Landscape.generate(args[0], 10);
    new Map(land, 9, "Testing Map");
  }
}
