/*
 * peer.c:
 * Object representing a connected peer.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 */

static const char rcsid[] = "$Id: peer.c,v 1.4 2003/10/19 03:10:23 chris Exp $";

#include <assert.h>

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <sys/socket.h>

#include "peer.h"
#include "util.h"

/* struct peer
 * Peer object internals. */
struct peer {
    int p_fd;       /* socket */
    /* XXX HACK! we should use circular buffers for this, but for the moment
     * just take the inefficient and easy way out. */
    uint8_t *p_rdbuf, *p_rdptr;
    size_t p_rdbuf_len;
    void *p_userdata;
    int p_errno;
    char *p_desc;
};

/* available PEER
 * Returns the number of bytes which remain available in the buffer. */
#define available(p)        ((p)->p_rdbuf_len - ((p)->p_rdptr - (p)->p_rdbuf))

#define DEFAULT_BUFFER_LENGTH       256

/* peer_new FD DATA
 * Create a new peer from the given FD, associating with it the given user
 * DATA. */
peer peer_new(int fd, void *userdata) {
    peer p;
    int i = 1;
    
    assert(fd != -1);
    alloc_struct(peer, p);
    p->p_fd = fd;
    p->p_rdbuf = p->p_rdptr = xmalloc(p->p_rdbuf_len = DEFAULT_BUFFER_LENGTH);

    p->p_userdata = userdata;
    
    /* Switch off the Nagle algorithm, since we want to minimise latency. */
    if (setsockopt(fd, SOL_TCP, TCP_NODELAY, &i, sizeof i) == -1)
        return NULL;
    
    return p;
}

#ifndef SHUT_RDWR
#   define SHUT_RDWR 2 /* Single UNIX Specification should give us this. */
#endif

/* peer_delete PEER
 * Delete the PEER object, disconnecting the peer if necessary. */
void peer_delete(peer p) {
    if (!p)
        return;
    shutdown(p->p_fd, SHUT_RDWR);
    close(p->p_fd);
    xfree(p->p_rdbuf);
    xfree(p->p_desc);
    xfree(p);
}

/* peer_get_userdata PEER
 * peer_set_userdata PEER DATA
 * Set a pointer to some user DATA in PEER. */
void *peer_get_userdata(peer p) {
    return p->p_userdata;
}

void peer_set_userdata(peer p, void *userdata) {
    p->p_userdata = userdata;
}

/* peer_pre_select PEER N READFDS WRITEFDS EXCEPTFDS
 * Set up the various file descriptor sets for this peer prior to a select
 * call. */
void peer_pre_select(peer p, int *n, fd_set *rd, fd_set *wr, fd_set *ex) {
    if (p->p_fd + 1 > *n) *n = p->p_fd + 1;
    FD_SET(p->p_fd, rd);
}

/* peer_post_select PEER READFDS WRITEFDS EXCEPTFDS
 * Do any processing required after select has returned. Returns 1 on success,
 * 0 if the connection is closed, and -1 on error. */
int peer_post_select(peer p, fd_set *rd, fd_set *wr, fd_set *ex) {
    if (FD_ISSET(p->p_fd, rd)) {
        ssize_t n;
        size_t offs, newlen;

        FD_CLR(p->p_fd, rd); /* XXX nasty */

        /* Ensure that we have at least DEFAULT_BUFFER_LENGTH / 2 of space to
         * read into. */
        offs = p->p_rdptr - p->p_rdbuf;
        newlen = p->p_rdbuf_len;
        while ((newlen - offs) < DEFAULT_BUFFER_LENGTH / 2)
            newlen *= 2;
        if (newlen != p->p_rdbuf_len)
            p->p_rdptr = (p->p_rdbuf = xrealloc(p->p_rdbuf, p->p_rdbuf_len = newlen)) + offs;
            
        do
            n = read(p->p_fd, p->p_rdptr, available(p));
        while (n == -1 && errno == EINTR);

        if (n == -1) {
            p->p_errno = errno;
            return -1;
        } else if (n == 0)
            return 0;
        else {
            p->p_rdptr += n;
            return 1;
        }
    } else
        return 1;
}

/* peer_write PEER BUFFER LENGTH
 * Write LENGTH bytes from BUFFER to the PEER. Returns 0 on success or -1
 * on failure. */
int peer_write(peer p, const uint8_t *buf, size_t len) {
    while (len > 0) {
        ssize_t n;
        do
            n = write(p->p_fd, buf, len);
        while (n == -1 && errno == EINTR);
        if (n == -1) {
            p->p_errno = errno;
            return -1;
        } else {
            len -= n;
            buf += n;
        }
    }
    return 0;
}

/* peer_get_data PEER LENGTH
 * Return a pointer to data read from the PEER, saving the number of bytes
 * available in *LENGTH. */
uint8_t *peer_get_data(peer p, size_t *len) {
    *len = p->p_rdptr - p->p_rdbuf;
    return p->p_rdbuf;
}

/* peer_consume PEER LENGTH
 * Consume LENGTH bytes of data from the beginning of the read buffer. */
void peer_consume(peer p, const size_t len) {
    assert(len <= available(p));
    memmove(p->p_rdbuf, p->p_rdbuf + len, available(p) - len);
    p->p_rdptr -= len;
}

/* peer_get_address PEER ADDRESS LENGTH
 * Save in *ADDRESS the address of the PEER, of length *LENGTH. */
void peer_get_address(peer p, struct sockaddr *a, socklen_t *l) {
    getpeername(p->p_fd, a, l);
}

/* peer_error PEER
 * Return the errno value associated with the last error on PEER, or zero if
 * no error has occured. */
int peer_error(peer p) {
    return p->p_errno;
}

/* peer_string PEER
 * Return a descriptive string for this peer. */
char *peer_string(peer p) {
    if (!p->p_desc) {
        struct sockaddr_in sin;
        socklen_t l;
        l = sizeof sin;
        peer_get_address(p, (struct sockaddr*)&sin, &l);
        p->p_desc = xmalloc(32);
        sprintf(p->p_desc, "%s:%d", inet_ntoa(sin.sin_addr), (int)htons(sin.sin_port));
    }
    return p->p_desc;
}

