/*
 * protocol.h:
 * Protocol stuff for the JRider server.
 *
 * Copyright (c) 2003 Chris Lightfoot. All rights reserved.
 * Email: chris@ex-parrot.com; WWW: http://www.ex-parrot.com/~chris/
 *
 * $Id: protocol.h,v 1.2 2003/10/19 01:33:03 chris Exp $
 *
 */

#ifndef __PROTOCOL_H_ /* include guard */
#define __PROTOCOL_H_

#include <stdbool.h>
#include <stdint.h>

/*
 * The server records the keys being pressed by each player, and relays the
 * information to the players at regular intervals. The server also distributes
 * other information about the individual players, in particular which track
 * and game they are playing on, and the configuration of their cars. This
 * information is also relayed to the other clients. The largest number of
 * players possible is six.
 */

#define MAX_PLAYERS 6

/*
 * Client to server.
 */

/* 
 * During gameplay, the clients send information about changes in key state.
 * Each packet contains a number of bytes. Each byte is structured as follows:
 *
 * +---+---+---------+---------+
 * |-1-|-1-|----3----|----3----|
 * +---+---+---------+---------+
 *  |   |   |         |
 *  |   |   |         key number
 *  |   |   player number
 *  |   |   (0 to 5 incl.)
 *  |   extended
 *  key up/down
 *  1 = up
 *  0 = down
 *
 * Keys are identified as: */
enum key_number {
    key_left = 0,
    key_right = 1,
    key_accelerate = 2,
    key_brake = 3,
    key_change_gear = 4,
    key_reserved_1 = 5,
    key_reserved_2 = 6,
    key_escape = 7
};
#define MAX_KEYS 8
/*
 * If the extended bit is set, the lower 6 bits of the byte represent a packet
 * type, and the byte is followed by two bytes representing a word in network
 * order giving the length in bytes of the following packet. Client to server
 * extended messages may be of any length, including zero.
 *
 * +---+---+------------------+
 * |-1-|-1-|--------6---------|
 * +---+---+------------------+
 *  |   |   |
 *  |   |   packet type
 *  |   extended
 *  |   must be 1
 *  unused
 * 
 */

/* enum tx_type
 * Representation of transmission types. */
enum tx_type {
    tx_none = 0, tx_rear = 1, tx_front = 2, tx_four = 3
};

/*
 * Extended packet types: client to server
 */
enum cs_message_type {
/* ID  type                     payload
 * --- ------------------------ ----------------------------------------------
 * 0   chat message             String to be relayed to all players. */
        cs_chat_message = 0,

/* 1   list tracks              None; should be empty. */
        cs_list_tracks = 1,

/* 2   choose track             Name of track; if the track name is one of the
 *                              active tracks, the player is added to that
 *                              game; otherwise, a new game is created. */
        cs_choose_track = 2,

/* 3   claim player             None; should be empty. */
        cs_claim_player = 3,
 
/* 4   choose car parameters    Single byte specifying a player number,
 *                              followed by any number of bytes of opaque
 *                              data. */
        cs_choose_car_parameters = 4,

/* 5   send checksums           Arbitrary byte string representing opaque
 *                              checksum data. The server uses this to ensure
 *                              that each client has the same model of the
 *                              track and the same code. */
        cs_send_checksums = 5
};


/* struct cs_message
 * Server representation of messages from client to server. */
struct cs_message {
    bool csm_extended;
    bool csm_key_up;
    uint8_t csm_player_num;
    enum key_number csm_key;

    /* Extended messages. */
    enum cs_message_type csm_type;
    uint8_t *csm_string;
    size_t csm_string_len;
};

/*
 * Server to client.
 */

/* 
 * During gameplay, the server sends updated information about the keys
 * being pressed at each point. The information is structured in 32-bit
 * words in network order:
 *
 * +-+-+-----+-----+-----+-----+-----+-----+
 * |1|1|--5--|--5--|--5--|--5--|--5--|--5--|
 * +-+-+-----+-----+-----+-----+-----+-----+
 *  | | |               ...           |
 *  | | |                             player 0 keymask
 *  | | player 5 keymask
 *  | extended
 *  escape key
 *
 * A bit in each key mask is set if the corresponding key is depressed.
 * 
 * If the top bit (escape key) is set, this should cause all players to
 * restart.
 * 
 * If the extended bit is set, then the next 6 bits are interpreted as a packet
 * type, the lowest 16 bits as a packet length, and the lowest 8 bits as the
 * first byte of the encapsulated data. The length of a packet is never zero.
 *
 * +-+-+------+----------------+--------+
 * |1|1|-- 6--|-------16-------|--- 8---|
 * +-+-+------+----------------+--------+
 *  | | |      |                |
 *  | | |      |                first byte of packet
 *  | | |      packet length
 *  | | packet type
 *  | extended
 *  unused
 */

/*
 * Extended packet types: server to client
 */
enum sc_message_type {
/* ID  type                     payload
 * --- ------------------------ ----------------------------------------------
 * 0   chat message             String message. */
        sc_chat_message = 0,
        
/* 1   track info               Single byte, number of players, followed by
 *                              string, track name. */
        sc_track_info = 1,
        
/* 2   assign player            Single byte giving a player number. */
        sc_assign_player = 2,
        
/* 3   game full                String message, indicates that no player could
 *                              be assigned. */
        sc_game_full = 3,

/* 4   set car parameters       Byte followed by opaque data, as above. */
        sc_set_car_parameters = 4,

/* 5   time step                Two byte word in network order giving time in
 *                              milliseconds between server updates.  Default
 *                              is ten. */
        sc_time_step = 5,
/* 6   player leaves            One byte, player number. */
        sc_player_leaves = 6
};

/* struct sc_message
 * Server representation of messages from server to client. */
struct sc_message {
    bool scm_extended;
    bool scm_player_key_map[MAX_PLAYERS][MAX_KEYS]; /* true means down */

    /* Extended messages */
    enum sc_message_type scm_type;
    uint8_t *scm_string;           /* string, allocated with malloc */
    size_t scm_string_len;
    uint8_t scm_player_num;     /* player number or number of players */
    uint16_t scm_timestep;
};

/* packets.c */
uint8_t *scm_to_wire(const struct sc_message *msg, size_t *plen, uint8_t *buf, size_t *buflen);
void scm_dump(FILE *fp, const char *prefix, const struct sc_message *scm);
void scm_free(struct sc_message *msg);
int csm_from_wire(struct cs_message *csm, const uint8_t *buf, const size_t len);
void cs_dump(FILE *fp, const char *prefix, struct cs_message *csm);
void csm_free(struct cs_message *csm);
#endif /* __PROTOCOL_H_ */
